/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.container.orchestration.provider.impl;

import com.github.dockerjava.api.DockerClient;
import com.github.dockerjava.api.async.ResultCallback;
import com.github.dockerjava.api.command.CreateContainerCmd;
import com.github.dockerjava.api.command.InspectImageResponse;
import com.github.dockerjava.api.command.PullImageCmd;
import com.github.dockerjava.api.command.PullImageResultCallback;
import com.github.dockerjava.api.model.AuthConfig;
import com.github.dockerjava.api.model.Bind;
import com.github.dockerjava.api.model.Container;
import com.github.dockerjava.api.model.Device;
import com.github.dockerjava.api.model.DeviceRequest;
import com.github.dockerjava.api.model.ExposedPort;
import com.github.dockerjava.api.model.HostConfig;
import com.github.dockerjava.api.model.Image;
import com.github.dockerjava.api.model.InternetProtocol;
import com.github.dockerjava.api.model.LogConfig;
import com.github.dockerjava.api.model.Ports;
import com.github.dockerjava.api.model.PullResponseItem;
import com.github.dockerjava.api.model.RestartPolicy;
import com.github.dockerjava.api.model.Volume;
import com.github.dockerjava.core.DefaultDockerClientConfig;
import com.github.dockerjava.core.DockerClientConfig;
import com.github.dockerjava.core.DockerClientImpl;
import com.github.dockerjava.httpclient5.ApacheDockerHttpClient;
import com.github.dockerjava.transport.DockerHttpClient;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import org.eclipse.kura.KuraErrorCode;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.configuration.ConfigurableComponent;
import org.eclipse.kura.container.orchestration.ContainerConfiguration;
import org.eclipse.kura.container.orchestration.ContainerInstanceDescriptor;
import org.eclipse.kura.container.orchestration.ContainerOrchestrationService;
import org.eclipse.kura.container.orchestration.ContainerPort;
import org.eclipse.kura.container.orchestration.ContainerState;
import org.eclipse.kura.container.orchestration.ImageConfiguration;
import org.eclipse.kura.container.orchestration.ImageInstanceDescriptor;
import org.eclipse.kura.container.orchestration.PasswordRegistryCredentials;
import org.eclipse.kura.container.orchestration.PortInternetProtocol;
import org.eclipse.kura.container.orchestration.RegistryCredentials;
import org.eclipse.kura.container.orchestration.listener.ContainerOrchestrationServiceListener;
import org.eclipse.kura.container.orchestration.provider.impl.ContainerOrchestrationServiceOptions;
import org.eclipse.kura.container.orchestration.provider.impl.enforcement.AllowlistEnforcementMonitor;
import org.eclipse.kura.crypto.CryptoService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerOrchestrationServiceImpl
implements ConfigurableComponent,
ContainerOrchestrationService {
    private static final String PARAMETER_CANNOT_BE_NULL = "The provided parameter cannot be null";
    private static final String UNABLE_TO_CONNECT_TO_DOCKER_CLI = "Unable to connect to docker cli";
    private static final Logger logger = LoggerFactory.getLogger(ContainerOrchestrationServiceImpl.class);
    private static final String APP_ID = "org.eclipse.kura.container.orchestration.provider.ConfigurableDocker";
    private ContainerOrchestrationServiceOptions currentConfig;
    private final Set<ContainerOrchestrationServiceListener> dockerServiceListeners = new CopyOnWriteArraySet<ContainerOrchestrationServiceListener>();
    private final Set<FrameworkManagedContainer> frameworkManagedContainers = new CopyOnWriteArraySet<FrameworkManagedContainer>();
    private DockerClient dockerClient;
    private CryptoService cryptoService;
    private AllowlistEnforcementMonitor allowlistEnforcementMonitor;
    private Map<String, String> containerInstancesDigests = new HashMap<String, String>();

    public void setDockerClient(DockerClient dockerClient) {
        this.dockerClient = dockerClient;
    }

    public void setCryptoService(CryptoService cryptoService) {
        this.cryptoService = cryptoService;
    }

    public void activate(Map<String, Object> properties) {
        logger.info("Bundle {} is starting with config!", (Object)APP_ID);
        if (!Objects.isNull(properties)) {
            this.updated(properties);
        }
        logger.info("Bundle {} has started with config!", (Object)APP_ID);
    }

    public void deactivate() {
        logger.info("Bundle {} is stopping!", (Object)APP_ID);
        if (this.testConnection()) {
            this.disconnect();
        }
        logger.info("Bundle {} has stopped!", (Object)APP_ID);
    }

    public void updated(Map<String, Object> properties) {
        logger.info("Bundle {} is updating with config!", (Object)APP_ID);
        ContainerOrchestrationServiceOptions newProps = new ContainerOrchestrationServiceOptions(properties);
        if (!newProps.equals(this.currentConfig)) {
            this.currentConfig = newProps;
            logger.info("Connecting to docker ");
            if (!this.currentConfig.isEnabled()) {
                this.cleanUpDocker();
                return;
            }
            if (this.allowlistEnforcementMonitor != null) {
                this.closeEnforcementMonitor();
            }
            this.connect();
            if (!this.testConnection()) {
                logger.error("Could not connect to docker CLI.");
                return;
            }
            logger.info("Connection Successful");
            if (this.currentConfig.isEnforcementEnabled()) {
                try {
                    this.startEnforcementMonitor();
                }
                catch (Exception ex) {
                    logger.error("Error starting enforcement monitor. Due to {}", (Object)ex.getMessage());
                    this.closeEnforcementMonitor();
                    logger.warn("Enforcement won't be active.");
                }
                this.enforceAlreadyRunningContainer();
            }
        }
        logger.info("Bundle {} has updated with config!", (Object)APP_ID);
    }

    private void startEnforcementMonitor() {
        logger.info("Enforcement monitor starting...");
        this.allowlistEnforcementMonitor = (AllowlistEnforcementMonitor)this.dockerClient.eventsCmd().withEventFilter(new String[]{"start"}).exec((ResultCallback)new AllowlistEnforcementMonitor(this.currentConfig.getEnforcementAllowlist(), this));
        logger.info("Enforcement monitor starting...done.");
    }

    private void closeEnforcementMonitor() {
        if (this.allowlistEnforcementMonitor == null) {
            return;
        }
        try {
            logger.info("Enforcement monitor closing...");
            this.allowlistEnforcementMonitor.close();
            this.allowlistEnforcementMonitor.awaitCompletion(5L, TimeUnit.SECONDS);
            this.allowlistEnforcementMonitor = null;
            logger.info("Enforcement monitor closing...done.");
        }
        catch (InterruptedException ex) {
            logger.error("Waited too long to close enforcement monitor, stopping it...", (Throwable)ex);
            Thread.currentThread().interrupt();
        }
        catch (IOException ex) {
            logger.error("Failed to close enforcement monitor, stopping it...", (Throwable)ex);
        }
    }

    private void enforceAlreadyRunningContainer() {
        if (this.allowlistEnforcementMonitor == null) {
            logger.warn("Enforcement wasn't started. Check on running containers will not be performed.");
            return;
        }
        logger.info("Enforcement check on already running containers...");
        this.allowlistEnforcementMonitor.enforceAllowlistFor(this.listContainerDescriptors());
        logger.info("Enforcement check on already running containers...done");
    }

    public List<String> listContainersIds() {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        ArrayList<String> result = new ArrayList<String>();
        for (Container cont : containers) {
            result.add(cont.getId());
        }
        return result;
    }

    public List<ContainerInstanceDescriptor> listContainerDescriptors() {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        ArrayList<ContainerInstanceDescriptor> result = new ArrayList<ContainerInstanceDescriptor>();
        containers.forEach(container -> {
            boolean bl = result.add(ContainerInstanceDescriptor.builder().setContainerName(this.getContainerName((Container)container)).setContainerImage(this.getContainerTag((Container)container)).setContainerImageTag(this.getContainerVersion((Container)container)).setContainerID(container.getId()).setContainerPorts(this.parseContainerPortsList(container.getPorts())).setContainerState(this.convertDockerStateToFrameworkState(container.getState())).setFrameworkManaged(this.isFrameworkManaged((Container)container)).build());
        });
        return result;
    }

    private Boolean isFrameworkManaged(Container container) {
        String containerName = this.getContainerName(container);
        return this.frameworkManagedContainers.stream().anyMatch(c -> ((FrameworkManagedContainer)c).name.equals(containerName));
    }

    private String getContainerName(Container container) {
        return container.getNames()[0].replace("/", "");
    }

    private String getContainerVersion(Container container) {
        String version = "";
        String[] image = container.getImage().split(":");
        if (image.length > 1 && !image[0].startsWith("sha256")) {
            version = image[1];
        }
        return version;
    }

    private String getContainerTag(Container container) {
        String[] image = container.getImage().split(":");
        if (image[0].startsWith("sha256")) {
            return "none";
        }
        return image[0];
    }

    private List<ContainerPort> parseContainerPortsList(com.github.dockerjava.api.model.ContainerPort[] ports) {
        ArrayList<ContainerPort> kuraContainerPorts = new ArrayList<ContainerPort>();
        Arrays.asList(ports).stream().forEach(containerPort -> {
            String ipTest = containerPort.getIp();
            if (ipTest != null && (ipTest.equals("::") || ipTest.equals("0.0.0.0"))) {
                kuraContainerPorts.add(new ContainerPort(containerPort.getPrivatePort().intValue(), containerPort.getPublicPort().intValue(), this.parsePortInternetProtocol(containerPort.getType())));
            }
        });
        return kuraContainerPorts;
    }

    private PortInternetProtocol parsePortInternetProtocol(String dockerPortProtocol) {
        switch (dockerPortProtocol) {
            case "tcp": {
                return PortInternetProtocol.TCP;
            }
            case "udp": {
                return PortInternetProtocol.UDP;
            }
            case "sctp": {
                return PortInternetProtocol.SCTP;
            }
        }
        throw new IllegalStateException();
    }

    private ContainerState convertDockerStateToFrameworkState(String dockerState) {
        switch (dockerState.trim()) {
            case "created": {
                return ContainerState.INSTALLED;
            }
            case "restarting": {
                return ContainerState.INSTALLED;
            }
            case "running": {
                return ContainerState.ACTIVE;
            }
            case "paused": {
                return ContainerState.STOPPING;
            }
            case "exited": {
                return ContainerState.STOPPING;
            }
            case "dead": {
                return ContainerState.FAILED;
            }
        }
        return ContainerState.INSTALLED;
    }

    public Optional<String> getContainerIdByName(String name) {
        this.checkRequestEnv(name);
        List containers = (List)this.dockerClient.listContainersCmd().withShowAll(Boolean.valueOf(true)).exec();
        for (Container cont : containers) {
            String[] containerNames;
            String[] stringArray = containerNames = cont.getNames();
            int n = containerNames.length;
            int n2 = 0;
            while (n2 < n) {
                String containerName = stringArray[n2];
                if (containerName.equals("/" + name)) {
                    return Optional.of(cont.getId());
                }
                ++n2;
            }
        }
        return Optional.empty();
    }

    public void startContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            this.dockerClient.startContainerCmd(id).exec();
        }
        catch (Exception e) {
            logger.error("Could not start container {}. It could be already running or not exist at all. Caused by {}", (Object)id, (Object)e);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    public String startContainer(ContainerConfiguration container) throws KuraException, InterruptedException {
        String containerId;
        this.checkRequestEnv(container);
        logger.info("Starting {} Microservice", (Object)container.getContainerName());
        Optional<ContainerInstanceDescriptor> existingInstance = this.listContainerDescriptors().stream().filter(c -> c.getContainerName().equals(container.getContainerName())).findAny();
        if (existingInstance.isPresent()) {
            if (existingInstance.get().getContainerState() == ContainerState.ACTIVE) {
                logger.info("Found already existing running container");
                containerId = existingInstance.get().getContainerId();
            } else {
                logger.info("Found already exisiting not running container, recreating it..");
                containerId = existingInstance.get().getContainerId();
                this.deleteContainer(containerId);
                this.pullImage(container.getImageConfiguration());
                containerId = this.createContainer(container);
                this.startContainer(containerId);
            }
        } else {
            logger.info("Creating new container instance");
            this.pullImage(container.getImageConfiguration());
            containerId = this.createContainer(container);
            this.addContainerInstanceDigest(containerId, container.getEnforcementDigest());
            this.startContainer(containerId);
        }
        logger.info("Container Started Successfully");
        if (container.isFrameworkManaged()) {
            this.frameworkManagedContainers.add(new FrameworkManagedContainer(container.getContainerName(), containerId));
        }
        return containerId;
    }

    public void stopContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            if (this.listContainersIds().contains(id)) {
                this.dockerClient.stopContainerCmd(id).exec();
            }
        }
        catch (Exception e) {
            logger.error("Could not stop container {}. Caused by {}", (Object)id, (Object)e);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    public void deleteContainer(String id) throws KuraException {
        this.checkRequestEnv(id);
        try {
            if (this.listContainersIds().contains(id)) {
                this.dockerClient.removeContainerCmd(id).exec();
            }
            this.frameworkManagedContainers.removeIf(c -> id.equals(((FrameworkManagedContainer)c).id));
            this.removeContainerInstanceDigest(id);
        }
        catch (Exception e) {
            logger.error("Could not remove container {}. Caused by {}", (Object)id, (Object)e);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR);
        }
    }

    private void checkRequestEnv(Object parameter) {
        if (Objects.isNull(parameter)) {
            throw new IllegalArgumentException(PARAMETER_CANNOT_BE_NULL);
        }
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
    }

    public void registerListener(ContainerOrchestrationServiceListener dockerListener) {
        this.dockerServiceListeners.add(dockerListener);
    }

    public void unregisterListener(ContainerOrchestrationServiceListener dockerListener) {
        this.dockerServiceListeners.remove(dockerListener);
    }

    private void imagePullHelper(final String imageName, final String imageTag, int timeOutSeconds, Optional<RegistryCredentials> repositoryCredentials) throws InterruptedException, KuraException {
        logger.info("Attempting to pull image: {}.", (Object)imageName);
        PullImageCmd pullRequest = this.dockerClient.pullImageCmd(imageName).withTag(imageTag);
        if (repositoryCredentials.isPresent()) {
            this.doAuthenticate(repositoryCredentials.get(), pullRequest);
        }
        (pullRequest.exec((ResultCallback)new PullImageResultCallback(){

            public void onNext(PullResponseItem item) {
                super.onNext(item);
                ContainerOrchestrationServiceImpl.this.createLoggerMessageForContainerPull(item, imageName, imageTag);
            }
        })).awaitCompletion(timeOutSeconds, TimeUnit.SECONDS);
    }

    private void doAuthenticate(RegistryCredentials repositoryCredentials, PullImageCmd pullRequest) throws KuraException {
        if (!(repositoryCredentials instanceof PasswordRegistryCredentials)) {
            throw new KuraException(KuraErrorCode.BAD_REQUEST);
        }
        PasswordRegistryCredentials repositoryPasswordCredentials = (PasswordRegistryCredentials)repositoryCredentials;
        AuthConfig authConfig = new AuthConfig().withUsername(repositoryPasswordCredentials.getUsername()).withPassword(new String(this.cryptoService.decryptAes(repositoryPasswordCredentials.getPassword().getPassword())));
        Optional url = repositoryPasswordCredentials.getUrl();
        if (url.isPresent()) {
            logger.info("Attempting to sign into repo: {}", url.get());
            authConfig = authConfig.withRegistryAddress((String)url.get());
        }
        pullRequest.withAuthConfig(authConfig);
    }

    private void createLoggerMessageForContainerPull(PullResponseItem item, String imageName, String imageTag) {
        if (logger.isDebugEnabled()) {
            logger.debug("Pulling {}:{} Layer {}, State: {}", new Object[]{imageName, imageTag, item.getId(), item.getStatus()});
        }
        if (item.isErrorIndicated()) {
            logger.error("Unable To Pull image {}:{} because : {}", new Object[]{item.getErrorDetail(), imageName, imageTag});
        }
        if (item.isPullSuccessIndicated()) {
            logger.info("Image pull of {}:{}, Layer: {}, was successful", new Object[]{imageName, imageTag, item.getId()});
        }
    }

    private String createContainer(ContainerConfiguration containerDescription) throws KuraException {
        if (!this.testConnection()) {
            throw new IllegalStateException("failed to reach docker engine");
        }
        if (containerDescription == null) {
            throw new IllegalStateException("failed to create container, null containerImage passed");
        }
        String containerImageFullString = String.format("%s:%s", containerDescription.getImageConfiguration().getImageName(), containerDescription.getImageConfiguration().getImageTag());
        CreateContainerCmd commandBuilder = null;
        try {
            commandBuilder = this.dockerClient.createContainerCmd(containerImageFullString);
            if (containerDescription.getContainerName() != null) {
                commandBuilder = commandBuilder.withName(containerDescription.getContainerName());
            }
            HostConfig configuration = new HostConfig();
            commandBuilder = this.containerEnviromentVariablesHandler(containerDescription, commandBuilder);
            commandBuilder = this.containerEntrypointHandler(containerDescription, commandBuilder);
            configuration = this.containerVolumeMangamentHandler(containerDescription, configuration);
            configuration = this.containerDevicesHandler(containerDescription, configuration);
            if (containerDescription.getRestartOnFailure()) {
                configuration = configuration.withRestartPolicy(RestartPolicy.unlessStoppedRestart());
            }
            configuration = this.containerPortManagementHandler(containerDescription, configuration, commandBuilder);
            configuration = this.containerLogConfigurationHandler(containerDescription, configuration);
            configuration = this.containerNetworkConfigurationHandler(containerDescription, configuration);
            configuration = this.containerMemoryConfigurationHandler(containerDescription, configuration);
            configuration = this.containerCpusConfigurationHandler(containerDescription, configuration);
            configuration = this.containerGpusConfigurationHandler(containerDescription, configuration);
            configuration = this.containerRuntimeConfigurationHandler(containerDescription, configuration);
            if (containerDescription.isContainerPrivileged()) {
                configuration = configuration.withPrivileged(Boolean.valueOf(containerDescription.isContainerPrivileged()));
            }
            String string = commandBuilder.withHostConfig(configuration).exec().getId();
            return string;
        }
        catch (Exception e) {
            logger.error("Failed to create container", (Throwable)e);
            throw new KuraException(KuraErrorCode.PROCESS_EXECUTION_ERROR);
        }
        finally {
            if (!Objects.isNull(commandBuilder)) {
                commandBuilder.close();
            }
        }
    }

    private HostConfig containerLogConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        LogConfig.LoggingType lt;
        switch (containerDescription.getContainerLoggingType().toUpperCase().trim()) {
            case "NONE": {
                lt = LogConfig.LoggingType.NONE;
                break;
            }
            case "LOCAL": {
                lt = LogConfig.LoggingType.LOCAL;
                break;
            }
            case "ETWLOGS": {
                lt = LogConfig.LoggingType.ETWLOGS;
                break;
            }
            case "JSON_FILE": {
                lt = LogConfig.LoggingType.JSON_FILE;
                break;
            }
            case "SYSLOG": {
                lt = LogConfig.LoggingType.SYSLOG;
                break;
            }
            case "JOURNALD": {
                lt = LogConfig.LoggingType.JOURNALD;
                break;
            }
            case "GELF": {
                lt = LogConfig.LoggingType.GELF;
                break;
            }
            case "FLUENTD": {
                lt = LogConfig.LoggingType.FLUENTD;
                break;
            }
            case "AWSLOGS": {
                lt = LogConfig.LoggingType.AWSLOGS;
                break;
            }
            case "DB": {
                lt = LogConfig.LoggingType.DB;
                break;
            }
            case "SPLUNK": {
                lt = LogConfig.LoggingType.SPLUNK;
                break;
            }
            case "GCPLOGS": {
                lt = LogConfig.LoggingType.GCPLOGS;
                break;
            }
            case "LOKI": {
                lt = LogConfig.LoggingType.LOKI;
                break;
            }
            default: {
                lt = LogConfig.LoggingType.DEFAULT;
            }
        }
        LogConfig lc = new LogConfig(lt, containerDescription.getLoggerParameters());
        configuration.withLogConfig(lc);
        return configuration;
    }

    private HostConfig containerNetworkConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        Optional networkMode = containerDescription.getContainerNetworkConfiguration().getNetworkMode();
        if (networkMode.isPresent() && !((String)networkMode.get()).trim().isEmpty()) {
            configuration.withNetworkMode(((String)networkMode.get()).trim());
        }
        return configuration;
    }

    private HostConfig containerMemoryConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        Optional memory = containerDescription.getMemory();
        if (memory.isPresent()) {
            try {
                configuration.withMemory((Long)memory.get());
            }
            catch (NumberFormatException e) {
                logger.warn("Memory value {} not valid. Caused by {}", memory.get(), (Object)e);
            }
        }
        return configuration;
    }

    private HostConfig containerCpusConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        Optional cpus = containerDescription.getCpus();
        cpus.ifPresent(cpu -> {
            configuration.withCpuPeriod(Long.valueOf(100000L));
            configuration.withCpuQuota(Long.valueOf((long)(100000.0f * ((Float)cpus.get()).floatValue())));
        });
        return configuration;
    }

    private HostConfig containerGpusConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        Optional gpus = containerDescription.getGpus();
        gpus.ifPresent(gpu -> {
            HostConfig hostConfig2 = configuration.withDeviceRequests((List)ImmutableList.of((Object)new DeviceRequest().withDriver("nvidia").withCount(Integer.valueOf(gpu.equals("all") ? -1 : Integer.parseInt(gpu))).withCapabilities((List)ImmutableList.of((Object)ImmutableList.of((Object)"gpu")))));
        });
        return configuration;
    }

    private HostConfig containerRuntimeConfigurationHandler(ContainerConfiguration containerDescription, HostConfig configuration) {
        Optional runtime = containerDescription.getRuntime();
        runtime.ifPresent(arg_0 -> ((HostConfig)configuration).withRuntime(arg_0));
        return configuration;
    }

    private HostConfig containerPortManagementHandler(ContainerConfiguration containerConfiguration, HostConfig hostConfig, CreateContainerCmd commandBuilder) {
        LinkedList<ExposedPort> exposedPorts = new LinkedList<ExposedPort>();
        Ports portbindings = new Ports();
        if (containerConfiguration.getContainerPorts() != null && !containerConfiguration.getContainerPorts().isEmpty()) {
            List containerPorts = containerConfiguration.getContainerPorts();
            for (ContainerPort port : containerPorts) {
                InternetProtocol ipProtocol = InternetProtocol.TCP;
                if (port.getInternetProtocol() != null) {
                    try {
                        ipProtocol = InternetProtocol.parse((String)port.getInternetProtocol().toString());
                    }
                    catch (IllegalArgumentException illegalArgumentException) {
                        logger.warn("Invalid internet protocol: {}. Using TCP.", (Object)port.getInternetProtocol());
                    }
                }
                ExposedPort exposedPort = new ExposedPort(port.getInternalPort(), ipProtocol);
                exposedPorts.add(exposedPort);
                portbindings.bind(exposedPort, Ports.Binding.bindPort((int)port.getExternalPort()));
            }
            if (exposedPorts.size() != portbindings.getBindings().size()) {
                logger.error("portsExternal and portsInternal must have the same size: {}", (Object)containerConfiguration.getContainerName());
            }
        }
        hostConfig.withPortBindings(portbindings);
        commandBuilder.withExposedPorts(exposedPorts);
        return hostConfig;
    }

    private CreateContainerCmd containerEnviromentVariablesHandler(ContainerConfiguration containerDescription, CreateContainerCmd commandBuilder) {
        if (containerDescription.getContainerEnvVars().isEmpty()) {
            return commandBuilder;
        }
        if (containerDescription.getContainerEnvVars() != null && !containerDescription.getContainerEnvVars().isEmpty()) {
            LinkedList<String> formattedEnvVars = new LinkedList<String>();
            for (String env : containerDescription.getContainerEnvVars()) {
                if (env.trim().isEmpty()) continue;
                formattedEnvVars.add(env.trim());
            }
            commandBuilder = commandBuilder.withEnv(formattedEnvVars);
        }
        return commandBuilder;
    }

    private CreateContainerCmd containerEntrypointHandler(ContainerConfiguration containerDescription, CreateContainerCmd commandBuilder) {
        if (containerDescription.getEntryPoint().isEmpty() || containerDescription.getEntryPoint() == null) {
            return commandBuilder;
        }
        return commandBuilder.withEntrypoint(containerDescription.getEntryPoint());
    }

    private HostConfig containerVolumeMangamentHandler(ContainerConfiguration containerDescription, HostConfig hostConfiguration) {
        if (containerDescription.getContainerVolumes().isEmpty()) {
            return hostConfiguration;
        }
        LinkedList<Bind> bindsToAdd = new LinkedList<Bind>();
        if (containerDescription.getContainerVolumes() != null && !containerDescription.getContainerVolumes().isEmpty()) {
            for (Map.Entry element : containerDescription.getContainerVolumes().entrySet()) {
                if (((String)element.getKey()).isEmpty() || ((String)element.getValue()).isEmpty()) continue;
                Volume tempVolume = new Volume((String)element.getValue());
                Bind tempBind = new Bind((String)element.getKey(), tempVolume);
                bindsToAdd.add(tempBind);
            }
            hostConfiguration = hostConfiguration.withBinds(bindsToAdd);
        }
        return hostConfiguration;
    }

    private HostConfig containerDevicesHandler(ContainerConfiguration containerDescription, HostConfig hostConfiguration) {
        if (containerDescription.getContainerDevices().isEmpty()) {
            return hostConfiguration;
        }
        if (containerDescription.getContainerDevices() != null && !containerDescription.getContainerDevices().isEmpty()) {
            LinkedList<Device> deviceList = new LinkedList<Device>();
            for (String deviceString : containerDescription.getContainerDevices()) {
                deviceList.add(Device.parse((String)deviceString));
            }
            if (!deviceList.isEmpty()) {
                hostConfiguration = hostConfiguration.withDevices(deviceList);
            }
        }
        return hostConfiguration;
    }

    private boolean doesImageExist(String imageName, String imageTag) {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List images = (List)this.dockerClient.listImagesCmd().exec();
        String requiredImage = String.valueOf(imageName) + ":" + imageTag;
        for (Image image : images) {
            if (Objects.isNull(image.getRepoTags())) continue;
            String[] stringArray = image.getRepoTags();
            int n = stringArray.length;
            int n2 = 0;
            while (n2 < n) {
                String tag = stringArray[n2];
                if (tag.equals(requiredImage)) {
                    return true;
                }
                ++n2;
            }
        }
        return false;
    }

    private void cleanUpDocker() {
        if (this.allowlistEnforcementMonitor != null) {
            this.closeEnforcementMonitor();
        }
        if (this.testConnection()) {
            this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onDisabled);
            this.disconnect();
        }
    }

    private boolean connect() {
        if (this.currentConfig.getHostUrl() == null) {
            return false;
        }
        DefaultDockerClientConfig config = DefaultDockerClientConfig.createDefaultConfigBuilder().withDockerHost(this.currentConfig.getHostUrl()).build();
        ApacheDockerHttpClient httpClient = new ApacheDockerHttpClient.Builder().dockerHost(config.getDockerHost()).build();
        this.dockerClient = DockerClientImpl.getInstance((DockerClientConfig)config, (DockerHttpClient)httpClient);
        boolean connected = this.testConnection();
        if (connected) {
            this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onConnect);
        }
        return connected;
    }

    private void disconnect() {
        if (this.testConnection()) {
            try {
                this.dockerServiceListeners.forEach(ContainerOrchestrationServiceListener::onDisconnect);
                this.dockerClient.close();
            }
            catch (IOException e) {
                logger.error("Error disconnecting", (Throwable)e);
            }
        }
    }

    private boolean testConnection() {
        boolean canConnect = false;
        try {
            this.dockerClient.pingCmd().exec();
            canConnect = true;
        }
        catch (Exception exception) {
            canConnect = false;
        }
        return canConnect;
    }

    public void pullImage(ImageConfiguration imageConfig) throws KuraException, InterruptedException {
        if (Objects.isNull(imageConfig.getImageName()) || Objects.isNull(imageConfig.getImageTag()) || imageConfig.getimageDownloadTimeoutSeconds() < 0 || Objects.isNull(imageConfig.getRegistryCredentials())) {
            throw new IllegalArgumentException("Parameters cannot be null or negative");
        }
        boolean imageAvailableLocally = this.doesImageExist(imageConfig.getImageName(), imageConfig.getImageTag());
        if (!imageAvailableLocally) {
            try {
                this.imagePullHelper(imageConfig.getImageName(), imageConfig.getImageTag(), imageConfig.getimageDownloadTimeoutSeconds(), imageConfig.getRegistryCredentials());
            }
            catch (InterruptedException e) {
                throw e;
            }
            catch (Exception e) {
                logger.error("Cannot pull container. Caused by ", (Throwable)e);
                throw new KuraException(KuraErrorCode.IO_ERROR, new Object[]{"Unable to pull container"});
            }
        }
    }

    public void pullImage(String imageName, String imageTag, int timeOutSeconds, Optional<RegistryCredentials> registryCredentials) throws KuraException, InterruptedException {
        this.pullImage(new ImageConfiguration.ImageConfigurationBuilder().setImageName(imageName).setImageTag(imageTag).setImageDownloadTimeoutSeconds(timeOutSeconds).setRegistryCredentials(registryCredentials).build());
    }

    public List<ImageInstanceDescriptor> listImageInstanceDescriptors() {
        if (!this.testConnection()) {
            throw new IllegalStateException(UNABLE_TO_CONNECT_TO_DOCKER_CLI);
        }
        List images = (List)this.dockerClient.listImagesCmd().withShowAll(Boolean.valueOf(true)).exec();
        ArrayList<ImageInstanceDescriptor> result = new ArrayList<ImageInstanceDescriptor>();
        images.forEach(image -> {
            InspectImageResponse iir = this.dockerClient.inspectImageCmd(image.getId()).exec();
            ImageInstanceDescriptor.ImageInstanceDescriptorBuilder imageBuilder = ImageInstanceDescriptor.builder().setImageName(this.getImageName((Image)image)).setImageTag(this.getImageTag((Image)image)).setImageId(image.getId()).setImageAuthor(iir.getAuthor()).setImageArch(iir.getArch()).setimageSize(iir.getSize().longValue());
            if (image.getLabels() != null) {
                imageBuilder.setImageLabels(image.getLabels());
            }
            result.add(imageBuilder.build());
        });
        return result;
    }

    private String getImageName(Image image) {
        if (image.getRepoTags() == null || image.getRepoTags().length < 1) {
            return "";
        }
        return image.getRepoTags()[0].split(":")[0];
    }

    private String getImageTag(Image image) {
        if (image.getRepoTags() == null || image.getRepoTags().length < 1 || image.getRepoTags()[0].split(":").length < 2) {
            return "";
        }
        return image.getRepoTags()[0].split(":")[1];
    }

    public void deleteImage(String imageId) throws KuraException {
        this.checkRequestEnv(imageId);
        try {
            this.dockerClient.removeImageCmd(imageId).exec();
        }
        catch (Exception e) {
            logger.error("Could not remove image {}. Caused by {}", (Object)imageId, (Object)e);
            throw new KuraException(KuraErrorCode.OS_COMMAND_ERROR, new Object[]{"Delete Container Image", "500 (server error). Image is most likely in use by a container."});
        }
    }

    public Set<String> getImageDigestsByContainerId(String containerId) {
        HashSet<String> imageDigests = new HashSet<String>();
        String containerName = this.listContainerDescriptors().stream().filter(container -> container.getContainerId().equals(containerId)).findFirst().map(container -> container.getContainerName()).orElse(null);
        if (containerName == null) {
            return imageDigests;
        }
        ((List)this.dockerClient.listImagesCmd().withImageNameFilter(containerName).exec()).stream().forEach(image -> {
            List<String> digests = Arrays.asList(image.getRepoDigests());
            digests.stream().forEach(digest -> {
                boolean bl = imageDigests.add(digest.split("@")[1]);
            });
        });
        return imageDigests;
    }

    private void addContainerInstanceDigest(String containerId, Optional<String> containerInstanceDigest) {
        if (containerInstanceDigest.isPresent()) {
            logger.info("Container {} presented enforcement digest. Adding it to the digests allowlist: it will be used if the enforcement is enabled.", (Object)containerId);
            this.containerInstancesDigests.put(containerId, containerInstanceDigest.get());
        } else {
            logger.info("Container {} doesn't contain the enforcement digest. If enforcement is enabled, be sure that the digest is included in the Orchestration Service allowlist", (Object)containerId);
        }
    }

    private void removeContainerInstanceDigest(String containerId) {
        if (this.containerInstancesDigests.containsKey(containerId)) {
            this.containerInstancesDigests.remove(containerId);
            logger.info("Removed digest of container with ID {} from Container Instances Allowlist", (Object)containerId);
            this.enforceAlreadyRunningContainer();
        }
    }

    public Set<String> getContainerInstancesAllowlist() {
        return new HashSet<String>(this.containerInstancesDigests.values());
    }

    private static class FrameworkManagedContainer {
        private final String name;
        private final String id;

        public FrameworkManagedContainer(String name, String id) {
            this.name = name;
            this.id = id;
        }

        public int hashCode() {
            return Objects.hash(this.id, this.name);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            FrameworkManagedContainer other = (FrameworkManagedContainer)obj;
            return Objects.equals(this.id, other.id) && Objects.equals(this.name, other.name);
        }
    }
}

