/*******************************************************************************
 * Copyright (c) 2004 INRIA.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Freddy Allilaire (INRIA) - initial API and implementation
 *******************************************************************************/
package org.eclipse.m2m.atl.adt.debug.ui;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.IValueDetailListener;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.m2m.atl.adt.debug.AtlDebugPlugin;
import org.eclipse.m2m.atl.adt.debug.AtlDebugMessages;
import org.eclipse.m2m.atl.adt.debug.core.AtlBreakpoint;
import org.eclipse.m2m.atl.adt.debug.core.AtlDebugTarget;
import org.eclipse.m2m.atl.adt.debug.core.AtlStackFrame;
import org.eclipse.m2m.atl.adt.debug.core.AtlThread;
import org.eclipse.m2m.atl.adt.debug.core.AtlVariable;
import org.eclipse.m2m.atl.engine.vm.ATLVMPlugin;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorRegistry;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.FileEditorInput;


/**
 * @author Freddy Allilaire
 * A debug model presentation is responsible for providing labels, images,
 * and editors associated with debug elements in a specific debug model.
 *
 * To allow for an extensible configuration, IDebugModelPresentation interface defines
 * a setAttribute method.
 * 
 */

public class AtlDebugModelPresentation extends LabelProvider implements IDebugModelPresentation {

	protected static Logger logger = Logger.getLogger(ATLVMPlugin.LOGGER);

	static final URL BASE_URL = AtlDebugPlugin.getDefault().getBundle().getEntry("/");//$NON-NLS-1$
	static final String iconPath = "icons/";//$NON-NLS-1$

	// If you add a constant here, look in the class AtlVariable if the value is not already used
	public final static int BREAKPOINT = 5;
	private Map mapImage = null;
	
	private HashMap fAttributes= new HashMap(3);

	/**
	 * If the button DISPLAY VARIABLE TYPE NAMES in debugUI is pressed
	 * then the status of the variable DISPLAY_VARIABLE_TYPE_NAMES is changed
	 * 
	 * This method returns true if DISPLAY_VARIABLE_TYPE_NAMES is selected in debugUI
	 */
	private boolean isShowVariableTypeNames() {
		Boolean show= (Boolean) fAttributes.get(DISPLAY_VARIABLE_TYPE_NAMES);
		show= show == null ? Boolean.FALSE : show;
		return show.booleanValue();
	}
	
	/**
	 * Clients may define new presentation attributes. For example, a client may wish
	 * to define a "hexadecimal" property to display numeric values in hexadecimal. Implementations
	 * should honor the presentation attributes defined by this interface where possible,
	 * but do not need to honor presentation attributes defined by other clients.
	 * To access the debug model presentation for a debug view, clients should use
	 * DebugView#getPresentation(String).
	 * 
	 * @see org.eclipse.debug.ui.IDebugModelPresentation#setAttribute(java.lang.String, java.lang.Object)
	 */
	public void setAttribute(String attribute, Object value) {
		if (value == null) {
			return;
		}
		fAttributes.put(attribute, value);
	}

	/**
	 * This method returns an image from the path
	 * @param path
	 * @return
	 */
	private Image createImage(String path) {
		try {
			URL url = new URL(BASE_URL, path);
			return ImageDescriptor.createFromURL(url).createImage();
		}
		catch(MalformedURLException e) {}
		return null;
	}

	private void initMapImage() {
		mapImage = new HashMap();
		mapImage.put(new Integer(AtlVariable.ATTRIBUTE), null);
		mapImage.put(new Integer(AtlVariable.ELEMENT), null);
		mapImage.put(new Integer(AtlVariable.LOCALVARIABLE), null);
		mapImage.put(new Integer(AtlVariable.REFERENCE), null);
		mapImage.put(new Integer(AtlVariable.SUPERTYPE), null);
		mapImage.put(new Integer(BREAKPOINT), null);
	}

	
	/**
	 * This method returns the image associate to the type of the parameter
	 * 
	 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
	 */
	public Image getImage(Object item) {
		if (mapImage == null)
			initMapImage();
		
		if (item instanceof AtlVariable) {
			String imageName = null;
			AtlVariable atlVar = (AtlVariable)item;
//			String typeVar = null;
//			try {
//				typeVar = atlVar.getReferenceTypeName();
//			} catch (DebugException e) {
//				e.printStackTrace();
//			}
			switch (atlVar.getDescription()) {
				case AtlVariable.ATTRIBUTE : imageName = "attribute.gif"; break;//$NON-NLS-1$
				case AtlVariable.ELEMENT : imageName = "element.gif"; break;//$NON-NLS-1$
				case AtlVariable.LOCALVARIABLE : imageName = "localVariable.gif"; break;//$NON-NLS-1$
				case AtlVariable.REFERENCE : imageName = "reference.gif"; break;//$NON-NLS-1$
				case AtlVariable.SUPERTYPE : imageName = "supertype.gif"; break;//$NON-NLS-1$
				default : return null;
			}				
			if (mapImage.get(new Integer(atlVar.getDescription())) == null) {
				mapImage.put(new Integer(atlVar.getDescription()), createImage(iconPath + imageName));
			}
			return (Image)mapImage.get(new Integer(atlVar.getDescription()));
		}
		else if (item instanceof AtlBreakpoint) {
			if (mapImage.get(new Integer(BREAKPOINT)) == null) {
				mapImage.put(new Integer(BREAKPOINT), createImage(iconPath + "breakpoint.gif"));//$NON-NLS-1$
			}
			return (Image)mapImage.get(new Integer(BREAKPOINT));
		}
		else if (item instanceof IMarker) {
			if (mapImage.get(new Integer(BREAKPOINT)) == null) {
				mapImage.put(new Integer(BREAKPOINT), createImage(iconPath + "breakpoint.gif"));//$NON-NLS-1$
			}
			return (Image)mapImage.get(new Integer(BREAKPOINT));
		}
		return null;
	}

	/**
	 * This method returns the text associate to the parameter
	 * 
	 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
	 */
	public String getText(Object item) {
		if (item instanceof AtlDebugTarget) {
			AtlDebugTarget target = (AtlDebugTarget) item;
			String name = "";//$NON-NLS-1$
			try {
				name = target.getName();
			}
			catch (DebugException e) {
				logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
//				e.printStackTrace();
			}
			return name + AtlDebugMessages.getString("AtlDebugModelPresentation.CONNECTEDONHOST")+ target.getHost() + AtlDebugMessages.getString("AtlDebugModelPresentation.PORT") + target.getPort(); //$NON-NLS-1$ //$NON-NLS-2$
		}
		else if (item instanceof AtlThread) {
			AtlThread thread = (AtlThread) item;
			try {
				String currentState;
				String message = "";//$NON-NLS-1$
				switch (((AtlDebugTarget)thread.getDebugTarget()).getState()) {
					case AtlDebugTarget.stateDisconnected: 	currentState = AtlDebugMessages.getString("AtlDebugModelPresentation.DISCONNECTED"); 	break; //$NON-NLS-1$
					case AtlDebugTarget.stateRunning: 		currentState = AtlDebugMessages.getString("AtlDebugModelPresentation.RUNNING"); 		break; //$NON-NLS-1$
					case AtlDebugTarget.stateSuspended: 	currentState = AtlDebugMessages.getString("AtlDebugModelPresentation.SUSPENDED"); //$NON-NLS-1$
															message = ((AtlDebugTarget)thread.getDebugTarget()).getMessageFromDebuggee();
															return thread.getName() + " (" + currentState + " (" + message + "))"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					case AtlDebugTarget.stateTerminated: 	currentState = AtlDebugMessages.getString("AtlDebugModelPresentation.TERMINATED"); 	break; //$NON-NLS-1$
					default : currentState = AtlDebugMessages.getString("AtlDebugModelPresentation.UNKNOWN"); //$NON-NLS-1$
				}
				return thread.getName() + " (" + currentState + ")";//$NON-NLS-1$//$NON-NLS-2$
			}
			catch (DebugException e) {
				return null;
			}
		}
		else if (item instanceof AtlStackFrame)
			return null;
		else if (item instanceof AtlVariable) {
			AtlVariable atlVar = (AtlVariable)item;
			String typeVar = null;
			try {
				if (isShowVariableTypeNames())
					typeVar = atlVar.getReferenceTypeName();
				else
					typeVar = "";//$NON-NLS-1$
				
				String rtn = atlVar.getValue().getReferenceTypeName();
				if (rtn.equals("Boolean") || //$NON-NLS-1$
					rtn.equals("Integer") ||//$NON-NLS-1$
					rtn.equals("Real"))//$NON-NLS-1$
					return typeVar + " " + atlVar.getName() + " = " + atlVar.getValue().getValueString();//$NON-NLS-1$//$NON-NLS-2$
				else if (rtn.equals("String"))//$NON-NLS-1$
					return typeVar + " " + atlVar.getName() + " = '" + atlVar.getValue().getValueString() + "'";//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
				else if (rtn.equals("EnumLiteral"))//$NON-NLS-1$
					return typeVar + " " + atlVar.getName() + " = #" + atlVar.getValue().getValueString();//$NON-NLS-1$//$NON-NLS-2$
				else if (rtn.equals("Map Element"))//$NON-NLS-1$
					return atlVar.getName();
				else
					return typeVar + " " + atlVar.getName() + " = " + atlVar.getReferenceTypeName() + " (id = " + atlVar.getIdVariable() + ")";//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
			} catch (DebugException e) {
				logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
//				e.printStackTrace();
			}
		}
		else if (item instanceof AtlBreakpoint)	{
			IMarker marker = ((AtlBreakpoint)item).getMarker();
			String location;
			try {
				location = (String)marker.getResource().getName();
				Integer lineNumber = (Integer)marker.getAttribute(IMarker.LINE_NUMBER);
				Integer charStart = (Integer)marker.getAttribute(IMarker.CHAR_START);
				Integer charEnd = (Integer)marker.getAttribute(IMarker.CHAR_END);
				return location + " [line: " + lineNumber + //$NON-NLS-1$
				", charStart: " + charStart + //$NON-NLS-1$
				", charEnd: " + charEnd +//$NON-NLS-1$
				"]";//$NON-NLS-1$
			}
			catch (CoreException e) {
				logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
//				e.printStackTrace();
			}
		}

		return null;
	}

	/**
	 * @see org.eclipse.debug.ui.IDebugModelPresentation#computeDetail(org.eclipse.debug.core.model.IValue,
	 *      org.eclipse.debug.ui.IValueDetailListener)
	 */
	public void computeDetail(IValue value, IValueDetailListener listener) {
		try {
			listener.detailComputed(value,value.getValueString());
		}
		catch (DebugException e) {
			logger.log(Level.SEVERE, e.getLocalizedMessage(), e);
//			e.printStackTrace();
		}
	}
	
	private class DisassemblyEditorInput implements IStorageEditorInput {

		private IStorage contents;
		
		public DisassemblyEditorInput(IStorage contents) {
			this.contents = contents;
		}
		
		/**
		 * @see org.eclipse.ui.IEditorInput#exists()
		 */
		public boolean exists() {
			return true;
		}

		/**
		 * @see org.eclipse.ui.IEditorInput#getImageDescriptor()
		 */
		public ImageDescriptor getImageDescriptor() {
			return null;
		}

		/**
		 * @see org.eclipse.ui.IEditorInput#getName()
		 */
		public String getName() {
			return contents.getName();
		}

		/**
		 * @see org.eclipse.ui.IEditorInput#getPersistable()
		 */
		public IPersistableElement getPersistable() {
			return null;
		}

		/**
		 * @see org.eclipse.ui.IEditorInput#getToolTipText()
		 */
		public String getToolTipText() {
			return contents.getName();
		}

		/**
		 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
		 */
		public Object getAdapter(Class adapter) {
			return null;
		}

		/**
		 * @see org.eclipse.ui.IStorageEditorInput#getStorage()
		 */
		public IStorage getStorage() throws CoreException {
			return contents;
		}
	}

//	private IEditorPart dte;
	
	public IEditorInput getDisassemblyEditorInput(AtlStackFrame frame) {
		IEditorInput ret = null;

		ret = new DisassemblyEditorInput(frame.getDisassembled());

		return ret;
	}

	/**
	 * 
	 * @see org.eclipse.debug.ui.ISourcePresentation#getEditorInput(java.lang.Object)
	 */
	public IEditorInput getEditorInput(Object element) {
		if(element instanceof AtlStackFrame ) {
			AtlStackFrame frame = (AtlStackFrame) element;
			if(((AtlDebugTarget)frame.getDebugTarget()).isDisassemblyMode()) return getDisassemblyEditorInput(frame);
			return new FileEditorInput(frame.getSourcefile());
		}
		else if(element instanceof AtlBreakpoint) {
			IMarker marker = ((AtlBreakpoint)element).getMarker();
			IFile ifile = (IFile)marker.getResource();
			return new FileEditorInput(ifile);
		}
		return null;
	}

	/**
	 * @see org.eclipse.debug.ui.ISourcePresentation#getEditorId(org.eclipse.ui.IEditorInput,
	 *      java.lang.Object)
	 */
	public String getEditorId(IEditorInput input, Object element) {
		if(input instanceof DisassemblyEditorInput)
			return "org.eclipse.ui.DefaultTextEditor";//$NON-NLS-1$
		IEditorRegistry registry= PlatformUI.getWorkbench().getEditorRegistry();
		IEditorDescriptor descriptor= registry.getDefaultEditor(input.getName());
		if (descriptor != null) {
			return descriptor.getId();
		} else {
			return "org.eclipse.ui.DefaultTextEditor";//$NON-NLS-1$
		}
	}

}