"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.escapeForAngleBracketLink = escapeForAngleBracketLink;
exports.needsAngleBracketLink = needsAngleBracketLink;
exports.hasBalancedParens = hasBalancedParens;
exports.removeNewUriExtIfNeeded = removeNewUriExtIfNeeded;
exports.resolveInternalDocumentLink = resolveInternalDocumentLink;
const vscode_uri_1 = require("vscode-uri");
const config_1 = require("../config");
const workspace_1 = require("../workspace");
const path_1 = require("./path");
const schemes_1 = require("./schemes");
/**
 * Escapes special characters so that {@linkcode linkText} can be used in an angle bracket link.
 */
function escapeForAngleBracketLink(linkText) {
    return linkText.replace(/([<>])/g, '\\$1'); // CodeQL [SM02383] This escaping is done for text in an editor, not for rendered markdown.
}
/**
 * Checks if {@linkcode linkText} needs to be enclosed in angle brackets.
 */
function needsAngleBracketLink(linkText) {
    // Links with whitespace or control characters must be enclosed in brackets
    // eslint-disable-next-line no-control-regex
    if (linkText.startsWith('<') || /\s|[\u007F\u0000-\u001f]/.test(linkText)) {
        return true;
    }
    return !hasBalancedParens(linkText);
}
function hasBalancedParens(linkText) {
    // Check if the link has balanced parens
    if (!/[\(\)]/.test(linkText)) {
        return true;
    }
    let previousChar = '';
    let nestingCount = 0;
    for (const char of linkText) {
        if (char === '(' && previousChar !== '\\') {
            nestingCount++;
        }
        else if (char === ')' && previousChar !== '\\') {
            nestingCount--;
        }
        if (nestingCount < 0) {
            return false;
        }
        previousChar = char;
    }
    return nestingCount === 0;
}
/**
 * Removes the file extension from {@link newUri} based on {@link LsConfiguration} preference.
 */
function removeNewUriExtIfNeeded(config, originalHref, newUri) {
    if (shouldRemoveNewUriExt(config, originalHref, newUri)) {
        const editExt = vscode_uri_1.Utils.extname(newUri);
        return newUri.with({
            path: newUri.path.slice(0, newUri.path.length - editExt.length)
        });
    }
    return newUri;
}
function shouldRemoveNewUriExt(config, originalHref, newUri) {
    if (!(0, path_1.looksLikeMarkdownUri)(config, newUri)) {
        return false;
    }
    switch (config.preferredMdPathExtensionStyle) {
        case config_1.PreferredMdPathExtensionStyle.removeExtension:
            return true;
        case config_1.PreferredMdPathExtensionStyle.includeExtension:
            return false;
        case config_1.PreferredMdPathExtensionStyle.auto:
        case undefined:
            // If the original markdown link did not use a file extension, remove ours too
            return !vscode_uri_1.Utils.extname(originalHref.path);
    }
}
function resolveInternalDocumentLink(sourceDocUri, linkText, workspace) {
    // Assume it must be an relative or absolute file path
    // Use a fake scheme to avoid parse warnings
    const tempUri = vscode_uri_1.URI.parse(`vscode-resource:${linkText}`);
    const docUri = workspace.getContainingDocument?.(sourceDocUri)?.uri ?? sourceDocUri;
    let resourceUri;
    if (!tempUri.path) {
        // Looks like a fragment only link
        if (typeof tempUri.fragment !== 'string') {
            return undefined;
        }
        resourceUri = sourceDocUri;
    }
    else if (tempUri.path[0] === '/') {
        const root = (0, workspace_1.getWorkspaceFolder)(workspace, docUri);
        if (root) {
            resourceUri = vscode_uri_1.Utils.joinPath(root, tempUri.path);
        }
    }
    else {
        if (docUri.scheme === schemes_1.Schemes.untitled) {
            const root = (0, workspace_1.getWorkspaceFolder)(workspace, docUri);
            if (root) {
                resourceUri = vscode_uri_1.Utils.joinPath(root, tempUri.path);
            }
        }
        else {
            const base = vscode_uri_1.Utils.dirname(docUri);
            resourceUri = vscode_uri_1.Utils.joinPath(base, tempUri.path);
        }
    }
    if (!resourceUri) {
        return undefined;
    }
    return {
        resource: resourceUri,
        linkFragment: tempUri.fragment,
    };
}
//# sourceMappingURL=mdLinks.js.map